<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/koneksi.php';
require_once 'config/cart.php';

// Cek apakah user sudah login
if (!isset($_SESSION['id_user'])) {
    header('Location: auth/login.php');
    exit;
}

    // Validasi input POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: keranjang.php');
    exit;
}

// Validasi data POST yang dibutuhkan
$required_fields = ['nama_pemesan', 'email_pemesan', 'alamat_pemesan', 'total_harga', 'produk_ids', 'produk_qty'];
foreach ($required_fields as $field) {
    if (!isset($_POST[$field])) {
        $_SESSION['error_message'] = 'Data tidak lengkap!';
        header('Location: keranjang.php');
        exit;
    }
}

$db = new Koneksi();
$conn = $db->getConnection();

try {
    // Mulai transaksi database
    $conn->begin_transaction();

    // Ambil data dari form
    $id_user = $_SESSION['id_user'];
    $nama_pemesan = trim($_POST['nama_pemesan']);
    $email_pemesan = trim($_POST['email_pemesan']);
    $alamat_pemesan = trim($_POST['alamat_pemesan']);
    $telepon = trim($_POST['telepon']);
    $total_harga = floatval($_POST['total_harga']);
    $produk_ids = $_POST['produk_ids'] ?? [];
    $produk_qty = $_POST['produk_qty'] ?? [];

    // Validasi data
    if (empty($nama_pemesan) || empty($email_pemesan) || empty($alamat_pemesan) || empty($telepon)) {
        throw new Exception('Data pemesan tidak lengkap!');
    }

    if (empty($produk_ids) || empty($produk_qty)) {
        throw new Exception('Tidak ada produk yang dipilih!');
    }

    if (count($produk_ids) !== count($produk_qty)) {
        throw new Exception('Data produk tidak valid!');
    }

    // Insert ke tabel transaksi
    $sql_transaksi = "INSERT INTO transaksi (id_user, nama_pemesan, email_pemesan, alamat_pemesan, telepon, total_harga, status_pesanan, tanggal_pesan) 
                      VALUES (?, ?, ?, ?, ?, ?, 'menunggu', CURRENT_TIMESTAMP)";
    
    $stmt_transaksi = $conn->prepare($sql_transaksi);
    $stmt_transaksi->bind_param("isssid", $id_user, $nama_pemesan, $email_pemesan, $alamat_pemesan, $telepon, $total_harga);
    
    if (!$stmt_transaksi->execute()) {
        throw new Exception('Gagal menyimpan transaksi: ' . $stmt_transaksi->error);
    }

    // Dapatkan ID transaksi yang baru dibuat
    $id_transaksi = $conn->insert_id;

    // Insert detail transaksi untuk setiap produk
    $sql_detail = "INSERT INTO detail_transaksi (id_transaksi, id_produk, jumlah_produk, harga_saat_transaksi) 
                   VALUES (?, ?, ?, ?)";
    $stmt_detail = $conn->prepare($sql_detail);

    foreach ($produk_ids as $index => $id_produk) {
        $id_produk = intval($id_produk);
        $jumlah = intval($produk_qty[$index]);

        // Ambil harga produk saat ini
        $sql_produk = "SELECT harga FROM produk WHERE id_produk = ?";
        $stmt_produk = $conn->prepare($sql_produk);
        $stmt_produk->bind_param("i", $id_produk);
        $stmt_produk->execute();
        $result_produk = $stmt_produk->get_result();
        $produk = $result_produk->fetch_assoc();

        if (!$produk) {
            throw new Exception('Produk dengan ID ' . $id_produk . ' tidak ditemukan!');
        }

        $harga_saat_transaksi = floatval($produk['harga']);

        // Insert detail transaksi
        $stmt_detail->bind_param("iiid", $id_transaksi, $id_produk, $jumlah, $harga_saat_transaksi);
        
        if (!$stmt_detail->execute()) {
            throw new Exception('Gagal menyimpan detail transaksi: ' . $stmt_detail->error);
        }

        // Hapus produk dari keranjang
        if (isset($_SESSION['cart'][$id_produk])) {
            unset($_SESSION['cart'][$id_produk]);
        }
    }

    // Commit transaksi
    $conn->commit();

    // Set session success message
    $_SESSION['success_message'] = 'Pesanan berhasil dibuat! Pesanan Anda sedang menunggu konfirmasi admin.';
    $_SESSION['last_order_id'] = $id_transaksi;

    // Redirect ke halaman sukses/pesanan
    header('Location: pesanan_saya.php');
    exit;

} catch (Exception $e) {
    // Rollback jika terjadi error
    $conn->rollback();
    
    $_SESSION['error_message'] = 'Gagal membuat pesanan: ' . $e->getMessage();
    header('Location: checkout.php');
    exit;
}
?>