<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/koneksi.php';
require_once 'config/cart.php';

// Cek apakah user sudah login
if (!isset($_SESSION['id_user'])) {
    header('Location: auth/login.php');
    exit;
}

$db = new Koneksi();
$conn = $db->getConnection();

// Ambil cart dari session
$cart = $_SESSION['cart'] ?? [];
$cart_products = [];
$grand_total = 0;

if (!empty($cart)) {
    $ids = implode(',', array_map('intval', array_keys($cart)));

    $sql = "
        SELECT p.*, k.nama_kategori
        FROM produk p
        LEFT JOIN kategori k ON p.id_kategori = k.id_kategori
        WHERE p.id_produk IN ($ids)
    ";

    $result = $conn->query($sql);

    while ($row = $result->fetch_assoc()) {
        $row['qty'] = $cart[$row['id_produk']];
        $row['subtotal'] = $row['harga'] * $row['qty'];
        $grand_total += $row['subtotal'];
        $cart_products[] = $row;
    }
}

// Handle AJAX requests
if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
    header('Content-Type: application/json');
    
    // Proses hapus item dari keranjang
    if (isset($_POST['action']) && $_POST['action'] === 'remove_item') {
        $id_produk = intval($_POST['id_produk']);
        removeFromCart($id_produk);
        echo json_encode(['success' => true, 'message' => 'Item berhasil dihapus']);
        exit;
    }
    
    // Proses update quantity
    if (isset($_POST['action']) && $_POST['action'] === 'update_qty') {
        $id_produk = intval($_POST['id_produk']);
        $new_qty = intval($_POST['qty']);
        
        if ($new_qty > 0) {
            $_SESSION['cart'][$id_produk] = $new_qty;
            
            // Hitung subtotal baru
            $sql = "SELECT harga FROM produk WHERE id_produk = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param('i', $id_produk);
            $stmt->execute();
            $result = $stmt->get_result();
            $product = $result->fetch_assoc();
            
            $subtotal = $product['harga'] * $new_qty;
            
            echo json_encode([
                'success' => true, 
                'qty' => $new_qty,
                'subtotal' => $subtotal,
                'subtotal_formatted' => number_format($subtotal, 0, ',', '.')
            ]);
        } else {
            removeFromCart($id_produk);
            echo json_encode(['success' => true, 'removed' => true]);
        }
        exit;
    }
}
?>

<head>
    <title>Keranjang Belanja - Plantify</title>
    <?php include 'includes/tailwind.php'; ?>
    <style>
        /* Hapus panah default pada input number */
        input[type=number]::-webkit-inner-spin-button, 
        input[type=number]::-webkit-outer-spin-button { 
            -webkit-appearance: none; 
            margin: 0; 
        }
    </style>
</head>
<?php include 'includes/navbar.php'; ?>
<body class="bg-gray-50 text-gray-800 font-sans antialiased mt-20">
    

    <div class="container mx-auto px-4 py-12">
        <div class="flex items-center justify-between mb-8" data-aos="fade-down">
            <div>
                <h1 class="text-3xl font-bold text-gray-900">Keranjang Belanja</h1>
                <p class="text-gray-500 mt-1">Kelola produk yang ingin Anda beli</p>
            </div>
            <div class="text-emerald-600 font-medium bg-emerald-50 px-4 py-2 rounded-full">
                <i class="fas fa-shopping-bag mr-2"></i> <span id="cart-count"><?= count($cart_products) ?></span> Item
            </div>
        </div>

        <?php if (empty($cart_products)): ?>
            <div class="max-w-2xl mx-auto text-center py-16" data-aos="zoom-in">
                <div class="mb-6 relative inline-block">
                    <div class="absolute inset-0 bg-emerald-100 rounded-full blur-xl opacity-50 animate-pulse"></div>
                    <i class="fas fa-shopping-basket text-8xl text-emerald-500 relative z-10"></i>
                </div>
                <h2 class="text-2xl font-bold text-gray-800 mb-2">Keranjang Anda Kosong</h2>
                <p class="text-gray-500 mb-8 max-w-md mx-auto">Sepertinya Anda belum menambahkan tanaman apapun. Yuk, hijaukan rumahmu sekarang!</p>
                <a href="index.php" class="inline-flex items-center bg-emerald-600 text-white px-8 py-3 rounded-xl hover:bg-emerald-700 transition-all duration-300 shadow-lg hover:shadow-emerald-200 transform hover:-translate-y-1">
                    <i class="fas fa-arrow-left mr-2"></i> Mulai Belanja
                </a>
            </div>
        <?php else: ?>
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                
                <div class="lg:col-span-2 space-y-4">
                    <div class="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden" id="cart-items-container">
                        <div class="hidden md:grid grid-cols-12 gap-4 p-4 bg-gray-50 border-b border-gray-100 text-sm font-semibold text-gray-500">
                            <div class="col-span-6 pl-2">Produk</div>
                            <div class="col-span-3 text-center">Jumlah</div>
                            <div class="col-span-3 text-right pr-2">Total</div>
                        </div>

                        <?php foreach ($cart_products as $index => $p): ?>
                            <div class="group relative p-4 md:p-6 border-b border-gray-50 last:border-0 hover:bg-gray-50 transition-colors duration-200 cart-item" 
                                 data-aos="fade-up" 
                                 data-aos-delay="<?= $index * 100 ?>"
                                 data-id="<?= $p['id_produk'] ?>">
                                <div class="grid grid-cols-1 md:grid-cols-12 gap-6 items-center">
                                    
                                    <div class="col-span-12 md:col-span-6 flex items-center gap-4">
                                        <input type="checkbox" 
                                               class="product-checkbox w-5 h-5 text-emerald-600 rounded border-gray-300 focus:ring-emerald-500 cursor-pointer"
                                               data-id="<?= $p['id_produk'] ?>"
                                               data-price="<?= $p['harga'] ?>"
                                               data-qty="<?= $p['qty'] ?>"
                                               checked>
                                        
                                        <div class="relative w-24 h-24 rounded-xl overflow-hidden bg-gray-100 shadow-sm flex-shrink-0 group-hover:shadow-md transition-all">
                                            <?php if (!empty($p['gambar'])): ?>
                                                <img src="uploads/produk/<?= htmlspecialchars($p['gambar']) ?>" 
                                                     alt="<?= htmlspecialchars($p['nama_produk']) ?>"
                                                     class="w-full h-full object-cover">
                                            <?php else: ?>
                                                <div class="w-full h-full flex items-center justify-center text-gray-400">
                                                    <i class="fas fa-image fa-lg"></i>
                                                </div>
                                            <?php endif; ?>
                                        </div>

                                        <div>
                                            <span class="text-xs font-bold text-emerald-600 uppercase tracking-wide"><?= htmlspecialchars($p['nama_kategori']) ?></span>
                                            <h3 class="font-bold text-gray-800 text-lg leading-tight mb-1"><?= htmlspecialchars($p['nama_produk']) ?></h3>
                                            <p class="text-sm text-gray-500">Rp <?= number_format($p['harga'], 0, ',', '.') ?> / item</p>
                                        </div>
                                    </div>

                                    <div class="col-span-6 md:col-span-3 flex justify-start md:justify-center">
                                        <div class="flex items-center border border-gray-200 rounded-lg bg-white shadow-sm">
                                            <button type="button" 
                                                    class="btn-decrease w-8 h-8 flex items-center justify-center text-gray-500 hover:text-emerald-600 hover:bg-emerald-50 rounded-l-lg transition-colors"
                                                    data-id="<?= $p['id_produk'] ?>"
                                                    data-qty="<?= $p['qty'] ?>">
                                                <i class="fas fa-minus text-xs"></i>
                                            </button>
                                            
                                            <input type="text" 
                                                   value="<?= $p['qty'] ?>" 
                                                   readonly
                                                   class="qty-input w-10 text-center text-sm font-semibold text-gray-800 border-x border-gray-200 h-8 focus:outline-none"
                                                   data-id="<?= $p['id_produk'] ?>">
                                            
                                            <button type="button"
                                                    class="btn-increase w-8 h-8 flex items-center justify-center text-gray-500 hover:text-emerald-600 hover:bg-emerald-50 rounded-r-lg transition-colors"
                                                    data-id="<?= $p['id_produk'] ?>"
                                                    data-qty="<?= $p['qty'] ?>">
                                                <i class="fas fa-plus text-xs"></i>
                                            </button>
                                        </div>
                                    </div>

                                    <div class="col-span-6 md:col-span-3 flex flex-col md:items-end justify-center gap-2">
                                        <p class="font-bold text-lg text-emerald-700 item-subtotal" data-id="<?= $p['id_produk'] ?>">
                                            Rp <?= number_format($p['subtotal'], 0, ',', '.') ?>
                                        </p>
                                        <button type="button"
                                                class="btn-remove text-xs text-red-400 hover:text-red-600 flex items-center gap-1 transition-colors"
                                                data-id="<?= $p['id_produk'] ?>">
                                            <i class="fas fa-trash-alt"></i> Hapus
                                        </button>
                                    </div>

                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <a href="index.php" class="inline-flex items-center text-gray-500 hover:text-emerald-600 font-medium transition-colors">
                        <i class="fas fa-arrow-left mr-2"></i> Lanjut Belanja
                    </a>
                </div>

                <div class="lg:col-span-1">
                    <div class="bg-white rounded-2xl shadow-lg border border-gray-100 p-6 sticky top-24" data-aos="fade-left">
                        <h2 class="text-xl font-bold text-gray-800 mb-6 flex items-center">
                            Ringkasan Pesanan
                        </h2>
                        
                        <div class="space-y-4 mb-6">
                            <div class="flex justify-between text-gray-600">
                                <span>Total Item</span>
                                <span class="font-medium" id="selected-count"><?= count($cart_products) ?> Item</span>
                            </div>
                            <div class="flex justify-between text-gray-600">
                                <span>Subtotal</span>
                                <span class="font-medium" id="subtotal-display">Rp <?= number_format($grand_total, 0, ',', '.') ?></span>
                            </div>
                            <div class="border-t border-dashed border-gray-200 my-4"></div>
                            <div class="flex justify-between items-end">
                                <span class="text-gray-800 font-bold text-lg">Total Bayar</span>
                                <span id="total-display" class="text-2xl font-bold text-emerald-600">Rp <?= number_format($grand_total, 0, ',', '.') ?></span>
                            </div>
                            <p class="text-xs text-gray-400 text-right mt-1">*Belum termasuk ongkir</p>
                        </div>

                        <form action="checkout.php" method="POST" id="checkout-form">
                            <input type="hidden" name="selected_products" id="selected-products">
                            <button type="submit" 
                                    class="w-full bg-emerald-600 text-white py-4 rounded-xl font-bold text-lg hover:bg-emerald-700 shadow-lg shadow-emerald-200 hover:shadow-xl hover:-translate-y-1 transition-all duration-300 flex items-center justify-center gap-2">
                                <span>Checkout Sekarang</span>
                                <i class="fas fa-arrow-right"></i>
                            </button>
                        </form>
                        
                        <div class="mt-6 flex items-center justify-center gap-2 text-gray-400 text-sm">
                            <i class="fas fa-shield-alt"></i>
                            <span>Pembayaran Aman & Terpercaya</span>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script type="module" src="assets/js/keranjang.js"></script>
</body>
</html>